/*******************************************************************/
/*                                                                 */
/*                      ADOBE CONFIDENTIAL                         */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                     */
/*                                                                 */
/* Copyright 2004 Adobe Systems Incorporated                       */
/* All Rights Reserved.                                            */
/*                                                                 */
/* NOTICE:  All information contained herein is, and remains the   */
/* property of Adobe Systems Incorporated and its suppliers, if    */
/* any.  The intellectual and technical concepts contained         */
/* herein are proprietary to Adobe Systems Incorporated and its    */
/* suppliers and may be covered by U.S. and Foreign Patents,       */
/* patents in process, and are protected by trade secret or        */
/* copyright law.  Dissemination of this information or            */
/* reproduction of this material is strictly forbidden unless      */
/* prior written permission is obtained from Adobe Systems         */
/* Incorporated.                                                   */
/*                                                                 */
/*******************************************************************/


/*
**-----------------------------------------------------------------------------
** Effect File Variables
**-----------------------------------------------------------------------------
*/

uniform float4x4	gWorldViewProj : WorldViewProjection; // This matrix will be loaded by the application
uniform float4x4	gWorldViewInverse;
uniform float4x4	gWorldView;
uniform float4		gLightPosition;
uniform float4		gLightColor = float4(0.8,0.8,0.8,1.0);
uniform float		gCurlAngle;
uniform float		gAspectRatio;
uniform float		gAnchorX;
uniform float4		gEyePosition;
uniform float		gTranslateX;
uniform float		gTranslateY;

texture				gVideoTexture;


/*
**-----------------------------------------
**		Sampler States
**-----------------------------------------
*/
//incoming video texture
sampler Sampler = sampler_state
{
    Texture   = (gVideoTexture);
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

/*
**-----------------------------------------------------------------------------
** Vertex Definitions
**-----------------------------------------------------------------------------
** APP_OUTPUT is the structure in which we receive the vertices from the application
*/
struct APP_OUTPUT
{
    float3 mPosition	: POSITION;
	float3 mNormal		: NORMAL;
	float2 mTexCoord	: TEXCOORD0;
	
};

/* 
** Pixel Shader structure declaration (For Foreground mesh)
*/

struct VS_OUTPUT 
{
  float4 mHPosition		: POSITION;		// coord position in window
  float2 mTexcoord		: TEXCOORD0;	// wavy or fleck map texture coordinates
  float3 mLightVec		: TEXCOORD1;		// position of light relative to point
  float3 mHalfVec		: TEXCOORD2;		// Blinn halfangle
  float3 mNormal		: TEXCOORD3;
};

/* 
** Pixel Shader structure declaration (For Background Mesh)
*/
struct PLAIN_VS_OUTPUT 
{
	float4 mHPosition		: POSITION;		// coord position in window
	float2 mTexcoord		: TEXCOORD0;	// texture coordinates
};



/*
** TEMP_OUT is a temporary structure for the ripple/Cyl Curl function
*/
struct TEMP_OUT
{
	float4 mPosition	: POSITION;
	float3 mNormal		: NORMAL0;
	float3 mTangent;
	float3 mBinormal;
};

/*
**------------------------------------------------
**		Cylindrical Curl effect
**------------------------------------------------
*/
TEMP_OUT DoCylCurl(float3 position)
{
	TEMP_OUT returnVertex;
	float sinVal, cosVal, curlRadius;
	
	float3x3 forwardMat, InvMat;
	
	sincos( gCurlAngle, sinVal, cosVal );
	forwardMat = float3x3( float3(cosVal, sinVal, 0.0), float3(-sinVal, cosVal, 0.0), float3(0,0,1) );
	sincos( -gCurlAngle, sinVal, cosVal );
	InvMat = float3x3( float3(cosVal, sinVal, 0.0), float3(-sinVal, cosVal, 0.0), float3(0,0,1) );
	
	position = mul( forwardMat, position );
	returnVertex.mPosition.xyz = position;
	returnVertex.mPosition.w = 1.0f;
	
	returnVertex.mNormal = float3( 0.0f, 0.0f, 1.0f );
	returnVertex.mBinormal = float3( 0, 1.0, 0 );	
	
	if ( position.x > gAnchorX )
	{
		//radius = 2/pi - factor*(distance to anchor)
		curlRadius = 0.50 - 0.02*( position.x - gAnchorX );
		float angle = ( position.x - gAnchorX )/curlRadius; // dist/Radius of curl
		sincos( angle, sinVal, cosVal );
		returnVertex.mPosition.x = gAnchorX + sinVal*curlRadius; // gAnchorX + R*SinVal
		returnVertex.mNormal = float3( -sinVal, 0, abs(cosVal) );
		returnVertex.mPosition.z = -0.50 + curlRadius*cosVal;
	}
	
	returnVertex.mPosition.xyz = mul( InvMat, returnVertex.mPosition.xyz );
	returnVertex.mBinormal = mul( InvMat, returnVertex.mBinormal );
	returnVertex.mNormal = mul( InvMat, returnVertex.mNormal );
	returnVertex.mTangent = cross( returnVertex.mBinormal, returnVertex.mNormal );
	
	return returnVertex;
}


/*
**-------------------------------------------------------------------------
** PageCurl Transition effect - Vertex Shader (For ForeGround Mesh)
**-------------------------------------------------------------------------
*/
VS_OUTPUT pagecurl_transition_vs(APP_OUTPUT In)
{
    VS_OUTPUT Out;

    // copy texture coordinates
	Out.mTexcoord.xy = In.mTexCoord.xy;
    
    TEMP_OUT tempVertex = DoCylCurl(float3(In.mPosition.x*gAspectRatio ,In.mPosition.yz));
	
	// transform vertex position into homogenous clip-space
    Out.mHPosition = mul(gWorldViewProj, tempVertex.mPosition);
    
	//Translating by a factor to take care of DirectX error which shifts by half pixel
	Out.mHPosition.xy -= float2( gTranslateX, -gTranslateY );
		
	// store light vector
    Out.mLightVec = gLightPosition.xyz;
   	
	//compute the half vector    
    float4 eyePos = float4(0.0, 0.0, 1.5, 0.0);
    Out.mHalfVec = normalize(Out.mLightVec + eyePos);
	Out.mNormal = tempVertex.mNormal;
	
	Out.mHalfVec.z = dot( tempVertex.mNormal, Out.mHalfVec );

	
	return Out;
}
/*
**-------------------------------------------------------------------------
** PageCurl Transition effect - pixel Shader 1_3 (For ForeGround Mesh)
**-------------------------------------------------------------------------
*/

float4 pagecurl_transition_ps_1_3(VS_OUTPUT In) : COLOR
{   
	float4 outColor, color1;
	float diffuse, specular;
	
	color1 = tex2D( Sampler, In.mTexcoord );
	
	diffuse = dot ( In.mNormal, In.mLightVec );
	specular = In.mHalfVec.z;
	specular *= specular;
	specular *= specular;
	specular *= specular;
	specular *= specular;
	
	outColor.xyz = color1*(diffuse) + (specular)*color1*float3(0.45,0.45,0.45);
	outColor.a = color1.a;
	
    return outColor;
}

/*
**-------------------------------------------------------------------------
** Page Curl Transition effect - Plain Vertex Shader(For Background Mesh)
**-------------------------------------------------------------------------
*/
PLAIN_VS_OUTPUT pagecurl_transition_vs_plain(APP_OUTPUT In)
{
	PLAIN_VS_OUTPUT Out;

	// copy texture coordinates
	Out.mTexcoord.xy = In.mTexCoord.xy;
	
	// transform vertex position into homogenous clip-space
	Out.mHPosition = mul(gWorldViewProj,float4(In.mPosition.x*gAspectRatio,In.mPosition.y,In.mPosition.z,1.0f));

	//Translating by a factor to take care of DirectX error which shifts by half pixel
	Out.mHPosition.xy -= float2( gTranslateX, -gTranslateY );
		
	return Out;
}


/*
**-------------------------------------------------------------------------
** Page Curl Transition effect - plain pixel Shader 1_3(For Background Mesh)
**-------------------------------------------------------------------------
*/

float4 pagecurl_transition_ps_1_3_plain(PLAIN_VS_OUTPUT In) : COLOR
{   
	float4 color1 = tex2D( Sampler, In.mTexcoord );
	return color1;
}

technique pagecurl_transition_1_3
{
    pass Pass0
    {
		//For Background Mesh (SourceB Video)
		Sampler[0] = (Sampler); 
		VertexShader = compile vs_1_1 pagecurl_transition_vs_plain();
		PixelShader  = compile ps_1_3 pagecurl_transition_ps_1_3_plain();
		
    }
    pass Pass1
    {
		//For Foreground Mesh (SourceA Video)
		Sampler[0] = (Sampler); 
		VertexShader = compile vs_1_1 pagecurl_transition_vs();
		PixelShader  = compile ps_1_3 pagecurl_transition_ps_1_3();
    }
}
